﻿using System;
using System.Collections.Generic;
using System.Xml.Linq;
using System.Windows.Forms;
using System.IO;

using XRE.Utils;

namespace XRE
{
	internal class XMLAccess
	{
		#region Constructor

		internal XMLAccess(string xmlFile = null)
		{
			string rootPath = Path.GetDirectoryName(Application.ExecutablePath);

            if(string.IsNullOrWhiteSpace(xmlFile)) {
				xmlFile = Path.Combine(rootPath, "XRE.xml");
			}

			if(!File.Exists(xmlFile)) {
				OpenFileDialog dialog = new OpenFileDialog();

				dialog.Filter = "XML Files (*.xml)|*.xml|All Files (*.*)|*.*";
				if(dialog.ShowDialog() == DialogResult.OK) {
					xmlFile = dialog.FileName;
				} else {
                    return;
				}
			}

			XElement xml = null;
			try {
				xml = XElement.Parse(File.ReadAllText(xmlFile));
				if(xml != null) {
					readSettings(xml);
					regAccess = new RegistryAccess(RootKeyName);
					readSections(xml.Element("Sections"), nodeList);
				}
			}
			catch(Exception e) {
				Debugging.ShowError(e.Message);
				return;
			}
		}

		#endregion

		#region Properties

		internal string ApplicationTitle
		{
			get
			{
				return applicationTitle;
			}
		}

		internal string Banner
		{
			get
			{
				return banner;
			}
		}

		internal List<XRENode> Nodes
		{
			get
			{
				return nodeList;
			}
		}

		internal bool HideDisclaimer
		{
			get
			{
				return hideDisclaimer;
			}
		}

		internal string RootKeyName {
			get
			{
				return rootKeyName;
			}
		}

		#endregion

		#region Public methods

		internal void WriteChangesToRegistry()
		{
			regAccess.WriteToRegistry(nodeList);
		}

		#endregion

		#region Private methods

		/// <summary>
		/// Read application settings from XML.
		/// </summary>
		/// <param name="xmlElement">XML element to read from.</param>
		private void readSettings(XElement xmlElement)
		{
			var xSettings = xmlElement.Element("Settings");

			applicationTitle = xSettings.GetElementValue("AppTitle");
			string path = xSettings.GetElementValue("ImagePath");
            imagePath = string.IsNullOrWhiteSpace(path) ? "" : Path.GetFullPath(path);
			banner = Path.Combine(imagePath, xSettings.GetElementValue("Banner"));
			rootKeyName = xSettings.GetElementValue("RegistryRoot");
			hideDisclaimer = xSettings.GetElementValue("HideDisclaimer").ToBoolean();
		}

		/// <summary>
		/// Read registry sections and items from XML.
		/// </summary>
		private void readSections(XElement xmlElement, List<XRENode> nodes)
		{
			foreach(var xKey in xmlElement.Elements("Section")) {

				// Add section

				var node = new XRENode(xKey.Attribute("name").Value, XRENode.NodeType.Section);
				nodes.Add(node);

				// Add items

				foreach(var xItem in xKey.Elements("Item")) {
					var entry = xItem.GetAttributeValue("entry");
					var name = xItem.GetAttributeValue("name");
					if(string.IsNullOrWhiteSpace(name)) {
						name = entry;
					}

					node.Nodes.Add(new XRENode(
						entry: entry,
                        type: XRENode.GetNodeType(xItem.GetAttributeValue("type")),
						name: name,
                        values: regAccess.GetValues(entry),

						imageName: Path.Combine(imagePath, xItem.GetAttributeValue("image")),
						hint: xItem.GetAttributeValue("hint"),
						width: xItem.GetAttributeValue("width"),
						options: xItem.GetAttributeValue("options"),
						unit: xItem.GetAttributeValue("unit"),

						minimum: xItem.GetAttributeValue("min", "null"),
						maximum: xItem.GetAttributeValue("max", "null"),
						increment: xItem.GetAttributeValue("incr", "1"),
						decimalPlaces: xItem.GetAttributeValue("places", "1"),
                        multiplier: xItem.GetAttributeValue("mult", "1")
                    ));
				}

				// Recursively look for subsections
				readSections(xKey, node.Nodes);
			}
		}

		#endregion

		#region Private fields

		private string applicationTitle;
		private string imagePath;
		private string banner;
		private string rootKeyName;
        private bool hideDisclaimer;
		private List<XRENode> nodeList = new List<XRENode>();
		private RegistryAccess regAccess;

		#endregion
	}
}
