﻿using System;
using System.Collections.Generic;
using Microsoft.Win32;

namespace XRE
{
	class RegistryAccess
	{
		#region Constructor

		internal RegistryAccess(string rootKeyName)
		{
			rootKey = getRegistryKey(rootKeyName);
			if(rootKey == null) {
				throw new Exception(string.Format("Registry root key {0} does not exist", rootKeyName));
			}
		}

		#endregion

		#region Public methods

		/// <summary>
		/// Return a list with string values from a comma-separated string.
		/// </summary>
		internal List<object> GetValues(string value)
		{
			var list = new List<object>();

			string[] parts = value.Split(',');
			foreach(string part in parts) {
				list.Add(GetValue(part));
			}
			return list;
		}

		/// <summary>
		/// Read the string value of an item.
		/// </summary>
		internal object GetValue(string key)
		{
			var split = key.Split(':');

			if(split.Length < 2) {
				return null;
			}

			string itemName = split[split.Length - 1];
			string subPath = key.Substring(0, key.Length - itemName.Length - 1).Replace(':', '\\');
			RegistryKey subKey = rootKey.OpenSubKey(subPath);

			if(subKey == null) {
				return null;
			}

			object value = subKey.GetValue(itemName);
			return value != null ? subKey.GetValue(itemName).ToString() : null;
		}

		/// <summary>
		/// Set the value of an item.
		/// </summary>
		internal void SetValue(string key, object value)
		{
			var split = key.Split(':');

			if(split.Length < 2) {
				return;
			}

			string itemName = split[split.Length - 1];
			string subPath = key.Substring(0, key.Length - itemName.Length - 1).Replace(':', '\\');
			RegistryKey subKey = rootKey.OpenSubKey(subPath, true);

			if(subKey == null) {
				return;
			}

            subKey.SetValue(itemName, value, subKey.GetValueKind(itemName));
		}

		/// <summary>
		/// Write values back to the registry.
		/// </summary>
		public void WriteToRegistry(List<XRENode> nodes)
		{
			foreach(XRENode node in nodes) {

				foreach(var subNode in node.Nodes) {
					if(subNode.Values != null) {
						string[] entries = subNode.Entry.Split(',');
						for(int i = 0; i < entries.Length; i++) {
							string entry = entries[i];
							SetValue(entry, subNode.Values[i]);
						}
					}
				}

				// Recursively look for subsections
				WriteToRegistry(node.Nodes);
			}
		}

		#endregion

		#region Private methods

		/// <summary>
		/// Transforms a key like "HKCU\Software\..." into a RegistryKey
		/// </summary>
		internal RegistryKey getRegistryKey(string fullKey)
		{
			var split = fullKey.Split('\\');
			string root = split[0];
			RegistryKey rootKey;

			switch(root) {
				case "HKEY_CURRENT_USER":
				case "HKCU":
					rootKey = Microsoft.Win32.Registry.CurrentUser;
					break;
				case "HKEY_LOCAL_MACHINE":
				case "HKLM":
					rootKey = Microsoft.Win32.Registry.LocalMachine;
					break;
				default:
					throw new Exception("Possible root values are HKCU and HKLM");
			}

			string remainder = fullKey.Substring(root.Length + 1);
			return rootKey.OpenSubKey(remainder);
		}

		#endregion

		#region Private fields

		private RegistryKey rootKey;

		#endregion
	}
}
