﻿using System;
using System.Windows.Forms;
using System.Drawing;

using XRE.Utils;
using System.Reflection;

namespace XRE.NodeControls
{
	internal class AngleControl : NodeControl
	{
		internal AngleControl(Control parent, XRENode xreNode) : base(parent, xreNode)
		{
			// Trackbar

			degrees = xreNode.Values[0].ToDecimal().ToDegrees();

			bool minIsNull = xreNode.Minimum == "null";
			bool maxIsNull = xreNode.Maximum == "null";
			multiplier = xreNode.Multiplier.ToDecimal(1);

			trackBar = new TrackBar();
			trackBar.AutoSize = false;
			trackBar.Bounds = new Rectangle(LabelEnd - 8, 2, DefaultControlWidth - labelWidth - 20, Bounds.Height - 5);
			trackBar.Minimum = minIsNull ? 0 : xreNode.Minimum.ToInt();
			trackBar.Maximum = maxIsNull ? 360 : xreNode.Maximum.ToInt();
			trackBar.SmallChange = xreNode.Increment.ToInt();
			trackBar.Value = (int)((int)degrees.Clamp((int)trackBar.Minimum, (int)trackBar.Maximum) * multiplier);
			trackBar.LargeChange = trackBar.SmallChange;
			trackBar.TickFrequency = Math.Max(xreNode.Increment.ToInt(), (trackBar.Maximum - trackBar.Minimum) / 12);
            trackBar.Scroll += new EventHandler(trackBarScroll);

			BasePanel.Controls.Add(trackBar);
			AssignToolTipTo(trackBar);

			// Label

			numericUpDown = new NumericUpDown();
			numericUpDown.Bounds = new Rectangle(trackBar.Right, 6, labelWidth, 20);
			numericUpDown.TextAlign = HorizontalAlignment.Right;
			numericUpDown.Minimum = trackBar.Minimum;
			numericUpDown.Maximum = trackBar.Maximum;
			numericUpDown.Increment = xreNode.Increment.ToInt();
            numericUpDown.Value = trackBar.Value;
			numericUpDown.DecimalPlaces = xreNode.DecimalPlaces.ToInt(0);
			numericUpDown.ThousandsSeparator = true;

			numericUpDown.TextChanged += new EventHandler(numericUpDown_TextChanged);

			xreNode.ChangedValue = trackBar.Value / multiplier;
			trackBar.ValueChanged += trackBar_ValueChanged;

			BasePanel.Controls.Add(numericUpDown);
			AssignToolTipTo(numericUpDown);

			// Force control double buffering

			BasePanel.GetType()
				.GetProperty("DoubleBuffered", BindingFlags.Instance | BindingFlags.NonPublic)
				.SetValue(BasePanel, true, null);

			BasePanel.Paint += basePanelPaint;
        }

		#region Private methods

		private void trackBar_ValueChanged(object sender, EventArgs e)
		{
			XreNode.ChangedValue = trackBar.Value / multiplier;
			XreNode.Values[0] = XreNode.ChangedValue.ToString();
		}

		private void numericUpDown_TextChanged(object sender, EventArgs e)
		{
			degrees = ((NumericUpDown)sender).Value;
			trackBar.Value = (int)degrees;
			BasePanel.Invalidate();
		}

		private void trackBarScroll(object sender, EventArgs e)
		{
			degrees = ((TrackBar)sender).Value;
            numericUpDown.Value = degrees;
			BasePanel.Invalidate();
		}

		private void basePanelPaint(object sender, PaintEventArgs e)
		{
			int radius = 12;
			const int centerX = 330;
			const int centerY = 16;

			// Draw circle and line

			e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
			e.Graphics.DrawEllipse(Pens.Gray, centerX - radius, centerY - radius,
				radius + radius, radius + radius);
			e.Graphics.DrawLine(Pens.DarkRed, centerX, centerY,
				centerX + (float)(radius * Math.Cos(((double)degrees).ToRadians())),
				centerY - (float)(radius * Math.Sin(((double)degrees).ToRadians()))
			);
		}

		#endregion

		#region Private fields

		private NumericUpDown numericUpDown;
		private TrackBar trackBar;
		private decimal multiplier = 1;
		private decimal degrees;
		private const int labelWidth = 40;

		#endregion
	}
}
