﻿using System;
using System.Drawing;
using System.Globalization;

namespace XRE.Utils
{
	/// <summary>
	/// Extension methods for converting strings into other types
	/// </summary>
	public static class Conversions
	{
		#region For strings

		public static bool ToBoolean(this string stringValue, bool defaultValue = false)
		{
			if(string.IsNullOrWhiteSpace(stringValue)) {
				return defaultValue;
			}

			stringValue = stringValue.ToLower();

			switch(stringValue) {
				case "true":
				case "1":
				case "on":
				case "yes":
					return true;

				case "false":
				case "0":
				case "off":
				case "no":
					return false;

				default:

					if(stringValue == bool.TrueString.ToLower()) {
						return true;
					} else if(stringValue == bool.FalseString.ToLower()) {
						return false;
					}

					int result;
					if(int.TryParse(stringValue, out result)) {
						if(result > 0) {
							return true;
						}
						return defaultValue;
					} else {
						return defaultValue;
					}
			}
		}

		public static int ToInt(this string stringValue, int defaultValue = 0)
		{
			int value;

			if(string.IsNullOrWhiteSpace(stringValue)) {
				return defaultValue;
			} else if(stringValue.ToLower().StartsWith("0x")) {
				stringValue = stringValue.Remove(0, 2);
			} else if(stringValue[0] == '#') {
				stringValue = stringValue.Remove(0, 1);
			} else {
				int.TryParse(stringValue, out value);
				return value;
			}
			int.TryParse(stringValue, NumberStyles.AllowHexSpecifier, null, out value);
			return value;
		}

		public static uint ToUInt(this string stringValue, uint defaultValue = 0)
		{
			uint value;

			if(string.IsNullOrWhiteSpace(stringValue)) {
				return defaultValue;
			} else if(stringValue.ToLower().StartsWith("0x")) {
				stringValue = stringValue.Remove(0, 2);
			} else if(stringValue[0] == '#') {
				stringValue = stringValue.Remove(0, 1);
			} else {
				uint.TryParse(stringValue, out value);
				return value;
			}
			uint.TryParse(stringValue, NumberStyles.AllowHexSpecifier, null, out value);
			return value;
		}

		public static long ToLong(this string stringValue, long defaultValue = 0)
		{
			long value;

			if(string.IsNullOrWhiteSpace(stringValue)) {
				return defaultValue;
			} else if(stringValue.ToLower().StartsWith("0x")) {
				stringValue = stringValue.Remove(0, 2);
			} else if(stringValue[0] == '#') {
				stringValue = stringValue.Remove(0, 1);
			} else {
				long.TryParse(stringValue, out value);
				return value;
			}
			long.TryParse(stringValue, NumberStyles.AllowHexSpecifier, null, out value);
			return value;
		}

		public static ulong ToULong(this string stringValue, ulong defaultValue = 0)
		{
			ulong value;

			if(string.IsNullOrWhiteSpace(stringValue)) {
				return defaultValue;
			} else if(stringValue.ToLower().StartsWith("0x")) {
				stringValue = stringValue.Remove(0, 2);
			} else if(stringValue[0] == '#') {
				stringValue = stringValue.Remove(0, 1);
			} else {
				ulong.TryParse(stringValue, out value);
				return value;
			}
			ulong.TryParse(stringValue, NumberStyles.AllowHexSpecifier, null, out value);
			return value;
		}

		public static decimal ToDecimal(this string stringValue, decimal defaultValue = 0)
		{
			decimal result;

			return decimal.TryParse(
					stringValue,
					NumberStyles.Float,
					new CultureInfo("en-US"),
					out result) ? result : defaultValue;
		}

		public static Color ToColor(this string stringValue)
		{
			return stringValue.ToColor(Color.Black);
		}

		public static Color ToColor(this string stringValue, Color defaultValue)
		{
			int colorValue;

			if(string.IsNullOrWhiteSpace(stringValue)) {
				return defaultValue;
			} else if(stringValue.ToLower().StartsWith("0x")) {
				stringValue = stringValue.Remove(0, 2);
			} else if(stringValue[0] == '#') {
				stringValue = stringValue.Remove(0, 1);
			} else {
				int.TryParse(stringValue, out colorValue);
				return Color.FromArgb(colorValue);
			}
			int.TryParse(stringValue, NumberStyles.AllowHexSpecifier, null, out colorValue);
			return Color.FromArgb(colorValue);
		}

		#endregion

		#region For integers

		public static string ToHex(this int intValue, int defaultValue = 0)
		{
			return string.Format("{0:X6}", intValue);
		}

		#endregion

		#region For floating-point values

		public static double ToDegrees(this double radians)
		{
			return radians * 180 / Math.PI;
		}

		public static double ToRadians(this double degrees)
		{
			return Math.PI / 180 * degrees;
		}

		public static decimal ToDegrees(this decimal radians)
		{
			return radians * 180 / (decimal)Math.PI;
		}

		public static decimal ToRadians(this decimal degrees)
		{
			return (decimal)Math.PI / 180 * degrees;
		}

		#endregion
	}
}
