﻿using System;
using System.Windows.Forms;
using System.Drawing;
using System.IO;
using XRE.Utils;

namespace XRE.NodeControls
{
	/// <summary>
	/// All node controls inherit from this class
	/// </summary>
	abstract class NodeControl
	{
		#region Constructor

		internal NodeControl(Control parent, XRENode xreNode)
		{
			this.xreNode = xreNode;

			// Base panel

			basePanel = new Panel();
			basePanel.Bounds = new Rectangle(0, yPos, parent.Width, panelHeight);

			// Separator line

			Label line = new Label();
			line.Bounds = new Rectangle(0, panelHeight - 1, parent.Width, 1);
			line.AutoSize = false;
			line.BackColor = Color.LightGray;
			basePanel.Controls.Add(line);

			// Tooltip and icon

			Label lblIcon = new Label();
			if(!string.IsNullOrWhiteSpace(xreNode.Hint)) {
				toolTip = xreNode.Hint;
				lblIcon.AutoSize = false;
				lblIcon.Bounds = new Rectangle(basePanel.Width - 16, 9, 16, 16);
				lblIcon.Image = Properties.Resources.Info;
				lblIcon.BringToFront();
				basePanel.Controls.Add(lblIcon);
				toolTip = xreNode.Hint;
				AssignToolTipTo(lblIcon);
				AssignToolTipTo(basePanel);
			}

			// Name label

			nameLabel = new Label();
			nameLabel.Bounds = new Rectangle(0, 6, labelWidth, 20);
			nameLabel.TextAlign = ContentAlignment.MiddleRight;
			nameLabel.Text = xreNode.Name;
			nameLabel.BackColor = Color.Transparent;
			AssignToolTipTo(nameLabel);
			basePanel.Controls.Add(nameLabel);

			// Image

			if(!string.IsNullOrWhiteSpace(xreNode.ImageName) && File.Exists(xreNode.ImageName)) {
				Label img = new Label();
				img.Image = new Bitmap(xreNode.ImageName);
				img.Bounds = new Rectangle(basePanel.Width - imageWidth, 2, imageWidth, imageHeight);
				basePanel.Controls.Add(img);
				AssignToolTipTo(img);
			}

			defaultControlWidth = basePanel.Width - imageWidth - labelEnd - rightPadding;

			parent.Controls.Add(basePanel);
		}

		#endregion

		#region Public methods

		internal static NodeControl Add(Control parent, XRENode xreNode, int top)
		{
			yPos = top;
			NodeControl newControl;

			switch(xreNode.Type) {

				// Scalar types

				case XRENode.NodeType.Integer:
				case XRENode.NodeType.UInt:
				case XRENode.NodeType.ULong:
				case XRENode.NodeType.Long:
				case XRENode.NodeType.Float:
					newControl = new NumericControl(parent, xreNode);
					break;

				// Other types

				case XRENode.NodeType.Boolean:
					newControl = new BooleanControl(parent, xreNode);
					break;
				case XRENode.NodeType.String:
					newControl = new StringControl(parent, xreNode);
					break;
				case XRENode.NodeType.Color:
					newControl = new ColorControl(parent, xreNode);
					break;
				case XRENode.NodeType.LOV:
					newControl = new LOVControl(parent, xreNode);
					break;
				default:
					Debugging.Report("Unknown node type: item \"{0}\"", xreNode.Name);
					return null;
			}

			return newControl;
		}

		internal void AssignToolTipTo(Control control)
		{
			if(!string.IsNullOrWhiteSpace(toolTip)) {
				ToolTip tt = new ToolTip();
				tt.SetToolTip(control, toolTip);
			}
		}

		#endregion

		#region Properties

		internal Panel BasePanel
		{
			get
			{
				return basePanel;
			}
		}

		internal Rectangle Bounds
		{
			get
			{
				return basePanel.Bounds;
			}
		}

		internal int LabelEnd
		{
			get
			{
				return labelEnd;
			}
		}

		internal int RightPadding
		{
			get
			{
				return rightPadding;
			}
		}

		internal int DefaultControlWidth
		{
			get
			{
				return defaultControlWidth;
			}
		}

		internal XRENode XreNode
		{
			get
			{
				return xreNode;
			}
		}

		#endregion

		#region Private fields

		private static int yPos = 0;
		private Label nameLabel;
		private Panel basePanel;
		private int defaultControlWidth;
		private string toolTip;
		private XRENode xreNode;

		// Geometric constants

		private const int labelWidth = 160;
		private const int labelEnd = labelWidth + 20;
		private const int rightPadding = 10;
		private const int panelHeight = imageHeight + 5;
		private const int imageWidth = 80;
		private const int imageHeight = 30;

		#endregion
	}
}