﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Drawing;
using System.Threading;

using XRE.NodeControls;
using XRE.Utils;

namespace XRE
{
	internal partial class MainForm : Form
	{
		#region Constructor

		internal MainForm(string[] args)
		{
#if !DEBUG
			Application.ThreadException += new ThreadExceptionEventHandler(Debugging.GenericExceptionHandler);
#endif
			InitializeComponent();
			this.Font = SystemInformation.MenuFont;

			Args = args;
		}

		#endregion

		#region Event handlers

		private void MainForm_Load(object sender, EventArgs e)
		{
			xmlAccess = new XMLAccess((Args.Length != 0 && !string.IsNullOrEmpty(Args[0])) ? Args[0] : "");

			if(xmlAccess.Nodes.Count == 0) {
				Close();
			}

#if !DEBUG
			if(!xmlAccess.HideDisclaimer) {
				showDisclaimer();
			}
#endif

			addIcon();
			addTitle();
            addLogo();

			// Adjust dimensions

			bottomOffset = this.Height - panel1.Height;
			panel1.MaximumSize = new Size(panel1.Width, this.MaximumSize.Height);

			// Add nodes to treeview

			addNodesToTree(treeView1.Nodes, xmlAccess.Nodes);
			if(treeView1.Nodes.Count > 0) {
				treeView1.SelectedNode = treeView1.Nodes[0];
			}

			linkLog.Visible = Debugging.Log.Length > 0;
		}

		private void MainForm_KeyDown(object sender, KeyEventArgs e)
		{
			if(e.KeyCode == Keys.F1) {
				btnHelp.PerformClick();
			}
		}

		private void treeView1_AfterSelect(object sender, TreeViewEventArgs e)
		{
			var selectedTreeNode = ((TreeView)sender).SelectedNode;
			lblSectionTitle.Text = selectedTreeNode.Text;

			// Add node controls
			
			panel1.Controls.Clear();

			int top = 0;

			foreach(XRENode xreNode in ((XRENode)selectedTreeNode.Tag).Nodes) {
				if(xreNode.Type != XRENode.NodeType.Section) {
					var ctrl = NodeControl.Add(panel1, xreNode, top);
					if(ctrl != null) {
						top = ctrl.Bounds.Bottom;
					}
				}
			}

			this.Height = Math.Max(this.Height, top + bottomOffset + yDelta);

			linkLog.Visible = Debugging.Log.Length > 0;
		}

		private void btnHelp_Click(object sender, EventArgs e)
		{
			Help.ShowHelp(this, "xre.chm");
		}

		private void btnApply_Click(object sender, EventArgs e)
		{
			xmlAccess.WriteChangesToRegistry();
		}

		private void linkLog_LinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
		{
			var form = new LogForm(Debugging.Log);
			form.ShowDialog();
		}

		#endregion

		#region Auxiliary methods

		private void showDisclaimer()
		{
			string disclaimer = "This software is a free utility that writes directly to the Windows registry. " +
				"Usage is entirely at your own risk. The author accepts no responsibility for " +
				"any undesirable effects or changes made to your computer. " + Environment.NewLine + Environment.NewLine +
				"Click OK to accept or Cancel to leave.";

            if(MessageBox.Show(this, disclaimer, "Important notice", 
				MessageBoxButtons.OKCancel, MessageBoxIcon.Warning) == DialogResult.Cancel) {
				this.Close();
				return;
			};
		}

		private void addTitle()
		{
			Text = string.Format(
				string.IsNullOrWhiteSpace(xmlAccess.ApplicationTitle) ? "{1} {2}" : "{0} ({1} {2})",
				xmlAccess.ApplicationTitle, Text, Application.ProductVersion);
		}

		private void addIcon()
		{
			if(string.IsNullOrWhiteSpace(xmlAccess.Icon)) {
				return;
			}
			if(!File.Exists(xmlAccess.Icon)) {
				Debugging.Report("Icon not found: {0}", xmlAccess.Icon);
				return;
			}
			this.Icon = new Icon(xmlAccess.Icon);
		}

		private void addLogo()
		{
			if(string.IsNullOrWhiteSpace(xmlAccess.Banner)) {
				return;
			}
			if(!File.Exists(xmlAccess.Banner)) {
				Debugging.Report("Banner not found: {0}", xmlAccess.Banner);
				return;
			}
            Label lblLogo = new Label();
			lblLogo.AutoSize = false;
			lblLogo.Image = new Bitmap(xmlAccess.Banner);
			int xOffset = treeView1.Left;
			int yOffset = treeView1.Top;
			lblLogo.Bounds = new Rectangle(xOffset, yOffset, 
				Math.Min(lblLogo.Image.Width, this.ClientRectangle.Width - xOffset * 2),
				lblLogo.Image.Height);
			this.Controls.Add(lblLogo);

			yDelta = lblLogo.Image.Height + yOffset;
			this.Height += yDelta;
			panel1.Top += yDelta;
			lblSectionTitle.Top += yDelta;
			lblSeparator.Top += yDelta;
			treeView1.Top += yDelta;
			treeView1.Height -= yDelta;
			this.MinimumSize = new Size(this.MinimumSize.Width, this.MinimumSize.Height + yDelta);
		}

		/// <summary>
		/// Recursively add XRENodes to a TreeView control.
		/// </summary>
		private void addNodesToTree(TreeNodeCollection treeNodes, List<XRENode> xreNodes)
		{
			foreach(XRENode xreNode in xreNodes) {
				if(xreNode.Type == XRENode.NodeType.Section) {
					TreeNode child = treeNodes.Add(xreNode.Name);
					addNodesToTree(child.Nodes, xreNode.Nodes);
					child.Tag = xreNode;
				}
			}
		}

		#endregion

		#region Private fields

		private int bottomOffset;
		private XMLAccess xmlAccess;
		private int yDelta = 0;
		private string[] Args;

		#endregion
	}
}
